package digest;

/**
 * Utility class to calculate the average value of a series of 
 * vectors, represented by arrays of ints and the standard deviation 
 * of the vectors.
 */
public class StatsFactory {
  private int vectorLength = 0;
  private long[] sum;
  private long sumsq;
  private int count=0;
  
  /**
   * Constructs an instance of StatsFactory ready to process vectors of 
   * the given length.
   * @param vectorLength The length of the vectors (length of the array) to expect.
   */
  public StatsFactory(int vectorLength){
    reset(vectorLength);
  }

  /**
   * Resets the factory to process a new series of vectors with the given length.
   * The running sums to calculate the average value and standard deviation 
   * are cleared.
   * @param vectorLength The length of the vectors (length of the array) to expect.
   */
  public void reset(int vectorLength){
    this.vectorLength = vectorLength;
    sum = new long[vectorLength];;
    sumsq = 0;
    count=0;
  }
  

  /**
   * Returns a vector of doubles that represent the average values of each element
   * of the processed vectors, that is, an average vector.
   */
  public double[] getAvg() {
    double[] avg = new double[vectorLength];
    for(int i=0; i<vectorLength; i++) {
      avg[i] = ((double)sum[i])/count;
    }
    return avg;
  }
  
  /**
   * Returns the standard deviation of the processed vectors.  This is the
   * average distance from the mean in the multi-dimensional space represented
   * by the vectors.
   */
  public double getStdDev() {
    double avg2 = 0.0;
    for(double x: getAvg()) {
      avg2 += x*x;
    }
    System.out.println("count = "+count+"  sumsq = "+sumsq+"  count*avg2 = "+count*avg2);
    return Math.sqrt((sumsq - count*avg2)/count);
  }
  
  /**
   * Add another vector in to be processed for the average and standard deviation.
   * To minimize round-off errors, the maximum value of any element of the vector
   * should be limited, e.g. to modulo 256.
   * @param nextVector  The vector to processed
   */
  public void add(int[] nextVector) {
    count++;
    for(int i=0; i < vectorLength; i++) {
      sum[i] += nextVector[i];
      sumsq += nextVector[i]*nextVector[i];
    }
  }

}