package lrs;

/**
 * Represents the abstract list state. Has a concrete toString () method that
 * uses anynomous inner classes to compute the String representation of the
 * LRStruct owner.
 * @author Dung X. Nguyen  Copyright 2005 - All rights reserved.
 * @since 02/09/05
 * @stereotype abstract state
 */
abstract class ANode {
    /**
     * Uses anonymous visitor class to compute a String representation.
     */
    String toString(LRStruct owner) {
        return (String)owner.execute (new IAlgo() {
            public Object emptyCase(LRStruct host, Object... inp) {
                return "()";
            }

            public Object nonEmptyCase(LRStruct host, Object... inp) {
                return "(" + host.getFirst() + host.getRest().execute(new IAlgo() {
                    public Object emptyCase(LRStruct h, Object... i) {
                        return ")";
                    }

                    public Object nonEmptyCase(LRStruct h, Object... i) {
                        return " " + h.getFirst() + h.getRest().execute (this);
                    }
                });
            }
        });
   }

    /**
    * Returns the tail LRStruct of the referencing LRStruct.
    * @param owner the LRStruct referencing this ANode.
    * @return the tail LRStruct of owner.
    * @throw java.util.NoSuchElementException if empty.
    */
    abstract LRStruct getRest(LRStruct owner);

    /**
    * Returns the first data object of the referencing LRStruct.
    * @param owner the LRStruct referencing this ANode.
    * @return the tail LRStruct of owner.
    * @throw java.util.NoSuchElementException if empty.
    */
    abstract Object getFirst(LRStruct owner);

    /**
    * Sets a new tail for the referencing LRStruct.
    * @param owner the LRS referencing this ANode.
    * @param tail the new tail for the owner LRStruct.
    * @throw java.util.NoSuchElementException if empty.
    * @return <code>LRStruct</code> owner
    */
    abstract LRStruct setRest(LRStruct owner, LRStruct tail);

    /**
    * Sets a new first data object for the referencing LRStruct.
    * @param owner the LRS referencing this ANode.
    * @param first the new data object for this ANode.
    * @throw java.util.NoSuchElementException if empty.
    * @return <code>LRStruct</code> owner
    */
    abstract LRStruct setFirst(LRStruct owner, Object dat);

    /**
    * Inserts a data object at the front of the LRStruct owner.
    * @param owner the LRS referencing this ANode.
    * @param dat the object to be inserted at the front.
    * @return <code>LRStruct</code> owner
    */
    abstract LRStruct insertFront(LRStruct owner, Object dat);

    /**
    * Removes and returns the first data object for the referencing LRStruct.
    * @param owner the LRS referencing this ANode.
    * @return the front data of the LRStruct owner.
    */
    abstract Object removeFront(LRStruct owner);

    /**
    * Executes a visitor algorithm and returns the output.
    * @param owner the LRS referencing this ANode.
    * @param algo the visitor algorithm to be executed.
    * @param inp the input needed by the algorithm.
    */
    abstract Object execute(LRStruct owner, IAlgo algo, Object... inp);
}

