package listFW.factory;

import listFW.*;

/**
 * Manufactures concrete IMTList and INEList objects.  Has only one
 * instance referenced by CompositeListFactory.Singleton.
 * MTList and NEList are static nested classes and hidden from all external
 * client code.  The implementations for MTList and NEList are the same as
 * before but completely invisible to the outside of this factory.
 * @author Dung X. Nguyen
 * @since Copyright 2004 by DXN - All rights reserved
 */
public class CompositeListFactory implements IListFactory {

    /**
     * Note the use of private static.
     */
    private static class MTList implements IMTList {
        public final static MTList Singleton = new MTList ();
        private MTList() {
        }

        final public Object execute(IListAlgo algo, Object inp) {
            return algo.emptyCase(this, inp);
        }

        public String toString() {
            return "()";
        }
    }

    /**
     * Note the use of private static.
     */
    private static class NEList implements INEList {
        private Object _first;
        private IList _rest;

        public NEList(Object dat, IList tail) {
            _first = dat;
            _rest = tail;
        }

        final public Object getFirst() {
            return _first;
        }

        final public IList getRest() {
            return _rest;
        }

        final public Object execute(IListAlgo algo, Object inp) {
            return algo.nonEmptyCase(this, inp);
        }


        public String toString() {
            return (String)ToStringAlgo.Singleton.nonEmptyCase(this, null);
        }
    }

    /**
     * Singleton Pattern
     */
    public static final CompositeListFactory Singleton = new CompositeListFactory();
    private CompositeListFactory() {
    }

    /**
     * Creates an empty list.
     * @return an IMTList object.
     */
    public IMTList makeEmptyList() {
        return MTList.Singleton;
    }

    /**
     * Creates a non-empty list containing a given first and a given rest.
     * @param first a data object.
     * @param tail != null, the rest of the non-empty list to be manufactured.
     * @return an INEList object containing first and tail
     */
    public INEList makeNEList(Object first, IList tail) {
        return new NEList(first, tail);
    }
}