package util;

public class SmartArray {
  /**
   * "Raw" array of data objects with initial capacity of 1.
   */
  private Object[] _data;
  
  /**
   * Index of the leftmost available empty slot in the _data array.
   * It is equal to the number of elements stored contigously in the _data array
   * starting with index 0.
   * Its default value at initialization is 0.
   * INVARIANT: we will always maintain _next <= _data.length.
   */
  private int _next;
  
  /**
   * Initializes _data to an array of one element.
   */
  public SmartArray() {
    _data = new Object[1];
  }
  
  /**
   * Intialize _data to a given source and _next to source.length.
   * NOTE: for privat use only!
   */
  private SmartArray(Object[] source) {
    _data = source;
    _next = source.length;
  }
  
  /**
   * Returns the number of elements stored in the _data array from index 0 to index _next - 1.
   */
  public int size() {
    return _next;
  }
  
  /**
   * Returns the element in _data array at index ix.
   * @throw ArrayOutOfBoundException if ix < 0 or ix >= _next.
   */
  public Object get(int ix) {
    checkBounds(ix);
    return _data[ix];
  }
  
  
  /**
   * Sets the _data array at the given index ix to the given element elt.
   * @param ix 0 <= ix <
   * Throws an ArrayIndexOutOfBoundsException if ix < 0 or ix >= _next.
   */
  public void set(int ix, Object elt) {
    checkBounds(ix);
    _data[ix] = elt;
  }
  
  /**
   * Removes and returns the data element at index ix.
   * Repacks _data and decrements _next;
   * Throws an ArrayIndexOutOfBoundsException if ix < 0 or ix >= _next.
   */
  public Object remove(int ix) {
    checkBounds(ix);
    Object result = _data[ix];
    _next--;
    for (int i = ix; i < _next ; i++) {
      _data[i] = _data[i+1];
    }
    return result;
  }
  
  /**
   * Add the given element object to the next available slot in the _data array.
   * If _next == _data.length, i.e. we have no more room in the array, 
   *   create a new array of twice the current legth,
   *   copy the _data array to the new array,
   *   assign the _data array to the new array.
   * Assign the given elt parameter to _data at _next.
   * Increment _next.
   */
  public void add(Object elt) {
    if (_data.length == _next) {  // there is no more room!
      Object[] newData = new Object[2 * _data.length];
      for (int i = 0; i < _data.length; i++) {
        newData[i] = _data[i]; 
      }
      _data = newData;
    }
    _data[_next] = elt;  
    _next++; // _next is the next available slot.
  }
  
  /**
   * Returns a new array containing all the elements in the _data array from index 0 to
   * index _next - 1, in the same order.
   */
  public Object[] toArray() {
    Object[] result = new Object[_next];
    for (int i = 0; i < _next; i++) {
      result[i] = _data[i];
    }
    return result;
  }
  
  /**
   * Returns a SmartArray that contains a copy of the given source "raw" array.
   */
  public static SmartArray toSmartArray(Object[] source) {
    Object[] data = new Object[source.length];
    for (int i = 0; i < source.length; i++) {
      data[i] = source[i];
    }
    return new SmartArray(data);        
  }
  
  /**
   * Returns a String representation that consists of a pair [_data, _next].
   * For example, [{12, 4, 5, 9, 3, 4, 77, 32}, 5]. 
   */
  public String toString() {
    return "[" + toString(_data) +", "+ _next + "]";
  }
  
  /**
   * Returns a String representation of an array as something like {a, b, c...}
   */
  public static String toString(Object[] elts) {
    String result = "{";
    if(elts.length>0) {
      for (int i = 0; i < elts.length - 1; i++) {
        result += elts[i] + ", ";
      }
      result += elts[elts.length - 1];
    }
    result +="}";
    return result;
  }
  
  /**
   * Removes all duplicated elements in this SmartArray. 
   * In the end, this SmartArray contains no duplicate elements.
   * Returns a SmartArray containing the elements that have duplicates in the original SmartArray.
   * The returned SmartArray does not contain any duplication.
   */
  public SmartArray remDup() {
    //TODO
    SmartArray result = new SmartArray();
    for (int i = 0; i < _next; i++) {
      boolean firstDup = true;
      for (int j = _next-1; j > i ; j--) {
        if (_data[j].equals(_data[i])) {
          Object temp = remove(j);
          if (firstDup) {
            result.add(temp);
            firstDup = false;
          }
        }
      }
    }
    return result;
  }  
  
  /**
   * Removes all duplicated elements in this SmartArray with fewer array traversals than remDup.
   * In the end, this SmartArray contains no duplicate elements.
   * Returns a SmartArray containing the elements that have duplicates in the original SmartArray.
   * The returned SmartArray does not contain any duplication.
   */
  public SmartArray remDupBetter() {
    // EXTRA CREDIT TODO
    SmartArray result = new SmartArray();
    for(int keyIdx=0; keyIdx<_next-1; keyIdx++) {  // Don't need to check last element, ok if you do.
      int shift = 0;
      for(int i=keyIdx+1; i<_next; i++) {
        if(_data[keyIdx].equals(_data[i])) {
          if(0==shift) result.add(_data[i]);
          shift++;
        }  
        else {
          _data[i-shift] = _data[i];
        }
      }
      _next -= shift;
    }
    return result;
  }  
  
  
  
  /**
   * Removes all the odd-indexed elements from this SmartArray and
   * returns them in an SmartArray.
   */
  public SmartArray unzip() {
    SmartArray result = new SmartArray();
    for(int i=1; i<_next;i++) { 
      result.add(remove(i));
    }
    return result;
  }

  /**
   * Removes all the odd-indexed elements from this SmartArray and
   * returns them in an SmartArray.
   */
  public SmartArray unzipBetter() {
    SmartArray result = new SmartArray();
    int shift = 0;
    for(int i=1; i<_next;i+=2) {
      result.add(_data[i]);
      shift++;
      if(i+1< _next) {
        _data[i+1-shift] = _data[i+1];
      }
    }
    _next -= shift;
    return result;
  }
  
  /**
   * Weaves another SmartArray into this SmartArray.
   * The other SmartArray remains unchanged.
   * This SmartArray is mutated to contain the elements of the other SmartArray at the odd-indexed position.
   */
  public void zip(SmartArray other) {
    Object[] newData = new Object[_next + other.size()];
    SmartArray smaller, larger;
    if(_next < other.size()) {
      smaller = this;
      larger = other;
    }
    else {
      smaller = other;
      larger = this;
    }
    
    for(int i=0; i< smaller.size();i++) {
      newData[2*i]= _data[i];
      newData[2*i+1]=other.get(i);
    }
    for(int i=smaller.size(), j=2*smaller.size();i<larger.size();i++, j++) {
      newData[j] = larger.get(i);
    }
    _data = newData;
    _next = _data.length;
  }
  
  /**
   * Checks if the supplied index is valid, i.e. 0<= ix < size()
   * throws an ArrayIndexOutOfBoundsException otherwise.
   * @throw ArrayOutOfBoundException if ix < 0 or ix >= size().
   * @param ix The index to check.
   */
  private void checkBounds(int ix) {
    if(ix < 0 || _next <= ix) {
      throw new ArrayIndexOutOfBoundsException("Attempt to access array at " + ix +
                                               " which is beyond the contiguous range [0.." +
                                               _next + "]");
    }
  }
  
}