package listFW.factory;

import listFW.*;

/**
 * Manufactures concrete IMTList and INEList objects.
 * Is a singleton.
 * Usese annonymous inner classes to implement IMTList and INEList.
 * @author Dung X. Nguyen
 * @author Stephen B. Wong
 * @since Copyright 2004 by DXN and SBW - All rights reserved
 */
public class InnerCompListFact implements IListFactory {
    /**
     * Singleton Pattern
     */
    public static final InnerCompListFact Singleton = new InnerCompListFact();
    private InnerCompListFact() {
    }
    
    private final static IListAlgo ToStrHelp = new IListAlgo() {
        /**
         * Returns the accumulated String + ")".
         * At end of list: done!  
         */
        public Object emptyCase(IMTList host, Object acc) {
            return  acc + ")";
        }
        
        /**
         * Continues accumulating the String representation by appending ", " + first to acc
         * and recur!
         */
        public Object nonEmptyCase(INEList host, Object acc) {
            return host.getRest().execute(this, acc + ", " + host.getFirst());
        }
    };
    
    private final static IMTList MTSingleton = new IMTList (){
        
        public Object execute(IListAlgo algo, Object inp) {
            return algo.emptyCase(this, inp);
        }
        
        public String toString() {
            return "()";
        }
    };
    
    /**
     * Creates an empty list.
     * @return an IMTList object.
     */
    public IMTList makeEmptyList() {
        return MTSingleton;
    }
    
    /**
     * Creates a non-empty list containing a given first and a given rest.
     * @param first a data object.
     * @param tail != null, the rest of the non-empty list to be manufactured.
     * @return an INEList object containing first and tail
     */
    public INEList makeNEList(final Object first, final IList tail) {
        return new INEList() {
            public Object getFirst() {
                return first;
            }
            
            public IList getRest() {
                return tail;
            }
            
            public Object execute(IListAlgo algo, Object inp) {
                return algo.nonEmptyCase(this, inp);
            }
                        
            public String toString() {
                return (String)tail.execute(ToStrHelp, "(" + first);
            }
        };
    }
}