
package brs.visitor;

import brs.*;
import java.util.*;

/**
* Deletes the given input from the host tree.  Returns TRUE if the input is in
* the host tree, otherwise return FALSE.
   <B>Invariant:</B> host contains unique objects and satisfies the BST property.
   <I>Post:</I> host does not contains the input.
   Algorithm:
   Case host is empty:
      returns FALSE because there is nothing to remove.
   Case host is not empty:
      if input < root
          return the result of asking the left subtree to delete the input;
      else if root < input
          return the result of asking the right subtree to delete the input;
      else if host's left subtree is empty  (at this point, input == root)
          ask host to remove its root (and become its right subtree)
          returns TRUE
      else {
         ask host to replace the root with the maximum value of its left subtree;
         the subtree that contains the maximum must have an empty right subtree;
         thus it is safe to ask this subtree to remove its root;
         return TRUE
      }
   <B>NOTE:</B>
   As you have been indoctrinated, it is "uncouth" do ask something for what it
   is. Instead of checking a subtree for emptiness, we should ask the subtree to
   help carry out the deletion.
* @author Dung X. Nguyen
* @since 03/07/03 Copyright 2003 - All rights reserved.
* @since 04/09/2004
*/
public class BSTDeleter implements IVisitor {
    private Comparator _order;

    /**
     * Used when the items in the tree are Comparable objects.
     */
    public BSTDeleter() {
        _order = new Comparator() {
            public int compare(Object x, Object y) {
                return ((Comparable)x).compareTo(y);
            }
        };
    }
    
    /**
     * Used when the items are ordered according to a given Comparator.
     * @param order a total ordering on the items stored in the tree.
     */
    public BSTDeleter (Comparator order) {
        _order = order;
    }

    /**
    * There is nothing to delete.
    * @param host is empty and obviously satisfies the BST Property.
    * @param input not used
    * @return Boolean.FALSE
    */
    public Object emptyCase(BiTree host, Object input) {
        return Boolean.FALSE;
    }

    /**
    if input < root
      ask the host's left subtree to delete the input;
    else if root < input
      ask the host's right subtree to delete the input
    else (at this point, input == root)
      ask the left subtree for help to remove the host's root using an
      anonymous helper.
    * @param host non-empty and satifies BST property
    * @param input object to be deleted from the host.
    * @return Boolean.
    */
    public Object nonEmptyCase(final BiTree host, Object input) {
        Object root = host.getRootDat();
        if (_order.compare(input, root) < 0) {
            return host.getLeftSubTree().execute (this, input);
        }
        if (_order.compare(input, root) > 0) {
            return host.getRightSubTree().execute (this, input);
        }
        // At this point:  input is equal to root.
        return host.getLeftSubTree().execute(new IVisitor() {
            /**
            * The outer host can safely remove its root and becomes its right
            * subtree.
            * @param h the left subtree of the outer host.
            */
            public Object emptyCase (BiTree h, Object notUsed) {
                host.remRoot();
                return Boolean.TRUE;
            }

            /**
            * Finds the maximum value in the h paramter.
            * Asks the outer host parameter to set its root to this maximum
            * value, and removes this maximum value from the subtree containing
            * this maximum value.
            * @param h the left subtree of the outer host.
            */
            public Object nonEmptyCase (BiTree h, Object notUsed) {
                BiTree maxTree = (BiTree)h.execute(MaxTreeFinder.Singleton, null);
                host.setRootDat(maxTree.remRoot());
                return Boolean.TRUE;
            }
        }, null);
    }
}


