package brs.visitor;
import brs.*;
import java.util.*;

/**
 * Finds a data object in a binary host tree that satisfies the
 * Binary Search Tree Property.
 * The algorithm is similar to that of insertion.
 * @author Dung X. Nguyen - Copyright 2004 - All Rights Reserved
 * @since 04/09/2004
 */
public class BSTFinder implements IVisitor {
    private Comparator _order;

    /**
     * Used when the items in the tree are Comparable objects.
     */
    public BSTFinder() {
        _order = new Comparator() {
            public int compare(Object x, Object y) {
                return ((Comparable)x).compareTo(y);
            }
        };
    }
    
    /**
     * Used when the items are ordered according to a given Comparator.
     * @param order a total ordering on the items stored in the tree.
     */
    public BSTFinder(Comparator order) {
        _order = order;
    }

    /**
    * Returns the empty host tree itself.
    * @param host an empty binary (which obviously satisfies the
    * Binary Search Tree Property).
    * @param input not used
    * @return BiTree the empty host tree.
    */
    public Object emptyCase(BiTree host, Object input) {
        return host;
    }

    /**
    * Returns the tree such that whose root, if it exists,  is equal to input
    * via the given Comparator _order.
    * @param host non empty and satisfies BSTP.
    * @param input object to be looked up.
    * @return BiTree
    */
    public Object nonEmptyCase(BiTree host, Object input) {
        Object root = host.getRootDat();
        if (_order.compare(input, root) < 0) {
            return host.getLeftSubTree().execute(this, input);
        }
        if (_order.compare(input, root) > 0) {
            return host.getRightSubTree().execute(this, input);
        }
        return host;
    }
}

