package brs.visitor;
import brs.*;
import java.util.*;

/**
* Inserts an Object into the host, maintaining the host's binary search tree
* property via a given Comparator.
* Can also be used for Comparable objects.
* Duplication is not allowed: replaces old data object with the new one.
* @author Dung X. Nguyen - Copyright 2003 - All rights reserved.
* @since 04/09/2004
*/
public class BSTInserter implements IVisitor {
    private Comparator _order;

    /**
     * Used when the items in the tree are Comparable objects.
     */
    public BSTInserter() {
        _order = new Comparator() {
            public int compare(Object x, Object y) {
                return ((Comparable)x).compareTo(y);
            }
        };
    }
    
    /**
     * Used when we want to order the items according to a given Comparator.
     * @param order a total ordering on the items to be stored in the tree.
     */
    public  BSTInserter (Comparator order) {
        _order = order;
    }

    /**
    * Returns the host tree where the input is inserted as the root.
    * @param host an empty binary tree, which obviously satisfies BSTP.
    * @param input new data to be inserted.
    * @return host (which is no longer empty).
    */
    public Object emptyCase(BiTree host, Object input) {
        host.insertRoot (input);
        return host;
    }

    /**
    * If the input is equal to host.getRootDat) then the old root data is
    * replaced by input.  Equality here is implicitly defined by the total
    * ordering represented by the Comparator _order.
    * @param host non-empty and satisfies BSTP.
    * @param input new data to be inserted.
    * @return the tree where input is inserted as the root.
    */
    public Object nonEmptyCase(BiTree host, Object input) {
        Object root = host.getRootDat();
        if (_order.compare(input, root) < 0) {
            return host.getLeftSubTree().execute(this, input);
        }
        if (_order.compare(input, root) > 0) {
            return host.getRightSubTree().execute(this, input);
        }
        host.setRootDat(input);
        return host;
    }
}

