package listFW.factory;

import listFW.*;

/**
 * Manufactures concrete IEmptyList and INEList objects.  Has only one
 * instance referenced by CompositeListFactory.Singleton<.
 * EmptyList and NEList are static nested classes and hidden from all external
 * client code.  The implementations for EmptyList and NEList are the same as
 * before but completely invisible to the outside of this factory.
 * @author Dung X. Nguyen
 * @since Copyright 2002 by DXN - All rights reserved
 */
public class CompositeListFactory implements IListFactory {

    /**
     * Note the use of private static.
     */
    private static class EmptyList implements IEmptyList {
        public final static EmptyList Singleton = new EmptyList ();
        private EmptyList() {
        }

        final public Object execute(IListAlgo algo, Object... inp) {
            return algo.emptyCase(this, inp);
        }
        
        public String toString() {
          return "()";
        }
    }

    /**
     * Note the use of private static.
     */
    private static class NEList implements INEList {
        private Object _first;
        private IList _rest;

        public NEList(Object dat, IList tail) {
            _first = dat;
            _rest = tail;
        }

        final public Object getFirst() {
            return _first;
        }

        final public IList getRest() {
            return _rest;
        }

        final public Object execute(IListAlgo algo, Object... inp) {
            return algo.nonEmptyCase(this, inp);
        }
        
        public String toString() {
          return "("+ _first + _rest.execute(new IListAlgo() {
            public Object emptyCase(IEmptyList host, Object...nu) {
              return ")";
            }
            public Object nonEmptyCase(INEList host, Object...nu) {
              return ", "+host.getFirst()+host.getRest().execute(this);
            }
          });
        }

    }

    /**
     * The unique instance of this class.
     */
    public static final CompositeListFactory Singleton
                        =  new CompositeListFactory();
    private CompositeListFactory() {
    }

    /**
     * Creates an empty list.
     * @return an IEmptyList object.
     */
    public IEmptyList makeEmptyList() {
        return EmptyList.Singleton;
    }

    /**
     * Creates a non-empty list containing a given first and a given rest.
     * @param first a data object.
     * @param tail != null, the rest of the non-empty list to be manufactured.
     * @return an INEList object containing first and tail
     * @exception IllegalArgumentException if tail == null.
     */
    public INEList makeNEList(Object first, IList tail) {
        if (null == tail) {
            throw new IllegalArgumentException("rest is null!");
        }
        return new NEList(first, tail);
    }
}