package lrs.visitor;
import lrs.*;

/**
 * Weaves ("Zips") together two distinct LRStructs that do not
 * share any node to yield a single LRStruct.
 * For example, when we zip (5, 3, 9), the host, together
 * with (-7, 0, 8, -15), the input parameter, both lists
 * become (5, -7, 3, 0, 9, 8, -15).
 */

public class Zipper implements IAlgo {
  public static final Zipper Singleton = new Zipper();
  private Zipper() {
  }
  
  
  /**
   * host is empty, so there is nothing to zip with.
   * Just let host become the input list.
   * @param host empty
   * @param input [0] the other LRStruct.
   */
  public Object emptyCase(LRStruct host, Object... input) {
    return host.execute(Share.Singleton, input[0]);
  }
  
  
  /**
   * Recursively weave the input LRStruct with host's rest.
   * The result is both input and host'rest share the same
   * head node.  All we need to do next is assign host to
   * input so that input "becomes" the host and share the same
   * head node as host.
   * @param host not empty
   * @param input [0] the other LRStruct.
   */
  public Object nonEmptyCase(LRStruct host, Object... input) {
    ((LRStruct)input[0]).execute(this, host.getRest());
    return ((LRStruct)input[0]).execute(Share.Singleton, host);
  }
}

