package kochModel.factory;

import kochModel.*;
import java.awt.geom.Point2D;
import lrs.*;

/**
 * This IAlgo constructs a list of Koch curves when it executed on a list of Point2D.Double prototype points.
 * It does so by Affine transforming (rotating, scaling and translating) the prototype points into the specified line segment.
 * This algorithm uses an AffineData object that carries the previous prototype point that was processed, plus an AffineXForm object
 * that was constructed using the Koch curve line segment being replaced.   When the algorithm is first executed on the prototype
 * list, it is assumed that the AffineData object carries a previous point that is  (0.0, 0.0).
 * @dependency kochModel.factory.AffineData uses
 */
class MakeManyAlgo implements IAlgo
{
	Point2D.Double endPoint = new Point2D.Double(1.0,0.0);

	/**
	 * This method is called when the end of the list of prototype points is encountered.
	 * The input parameter, param, is assumed to be a AffineData object, which stores the previous point processed and
	 * an AffineXForm object.    The enpoint property (1.0, 0,0) is used as the last prototype point.
	 * The AffineXForm object is used to transform the prototype interval into the curve's interval.
	 * A LRStruct with one Koch curve in it is generated, where the Koch curve is over the interval from the previous point to the endpoint.
	 *
	 *
	 * @param host The LRStruct that is the current node.
	 * @param param The AffineData object carrying the previous point and the AffineXForm object.
	 * @return A LRSTruct with one Koch curve in it.
	 */
	public Object emptyCase(LRStruct host, Object param)
	{
		AffineData ad = (AffineData)param;   // Convert the type of the input param.
		Koch k = makeKoch(ad,endPoint);     // Make a new Koch object with the supplied point and (1,0).
		LRStruct l = new LRStruct();        // Create a new list for Koch objects
		l.insertFront(k);            // Put the new Koch object in the list.
		return l;                    // return the list.
	}

	/**
	 * This method is called whenever a specified point of the prototype list is being processed (i.e. the list is not empty yet).
	 * A new base case Koch object is created using the Affine translated points from the prototype and inserted as first into the list
	 * returned by the recursive call to the rest of the prototype list, having replaced the previous point in the AffineData object with the present point..
	 * @param host The LRStruct that is the current node.
	 * @param param An AffineData object carrying important information.
	 * @return A LRStruct filled with Koch curves.
	 */
	public Object nonEmptyCase(LRStruct host, Object param)
	{
		AffineData ad = (AffineData)param;      // Convert the type of the input param.
		Koch k = makeKoch(ad,(Point2D.Double)host.getFirst());  // make a new Koch object with the supplied (previous_ point and the present point.
		ad.setX((Point2D.Double)host.getFirst());    // Save this point as the point to pass to the next node.
		LRStruct l = (LRStruct)(host.getRest().execute(this, ad));  // get list of Koch's made by rest of list
		l.insertFront(k);    // insert the new Koch into the list.
    		return l;          // return the list
	}

	/**
	 * A "helper" function for  nullCase() and nonNullCase() that will take the AffineData object and the present point and returns a new base case Koch object.
	 * @param ad The AffineData object needed.
	 * @param x The present point being processed.
	 * @return A new Koch object.
	 */
	private Koch makeKoch(AffineData ad, Point2D.Double x)
	{
		return new Koch(ad.doXForm(ad.getX()), ad.doXForm(x)); // Create a new Koch at the supplied points.
	}
}

