package infixModel;
/**
 * Represents the calculator itself, without its GUI and separate from its states.
 * Copyright (C) 2004 Dung X. Nguyen. All rights Reserved.
 */
public class InfixCalc {
    /**
     * accumulated computation
     */
    private double _acc; 
    /**
     * current display string
     */
    private String _display;
    /**
     * Current pending operation
     */
    private IBinOp _pendingOp; 
    /**
     * A no-op
     */
    private final IBinOp _noOp = new IBinOp() {
        public double compute(double n1, double n2) {
            return Double.parseDouble(_display);
        }
    };
    
    /**
     * The current state
     */
    private ACalcState _curState; 
    
    
    
    /**
     * Inialize this Calculator to the "clear" state.
     */
    public InfixCalc() {
        clear();
    }
    
    
    /**
     * Enters a digit.
     * @param c '0' <= c <= '9'.
     * @return accumulated input.
     */
    public final String enterDigit(char c) {
        System.out.println("Infix Calc enter digit " + c + " : pre display " + _display);
        _curState.enterDigit(this, c);
        System.out.println("Infix Calc enter digit: post display " + _display);
        return _display;
    }
    
    /**
     * Enters a decimal point
     */
    public final String enterPoint() {
        _curState.enterPoint(this);
        System.out.println("Infix Calc enter point: post display " + _display);
        return _display;
    }
    
    /**
     * Perform the pending operation, and prepare to add whatever is computed so
     * far to the operand that will be entered next and return the resulting String.
     * @param op The new pending operation.
     */
    public final String enterOp(IBinOp op) {
        _curState.enterOp(this, op);
        System.out.println("Infix Calc enter op: post display " + _display);
        return _display;
    }
    
    /**
     * Compute the current operation and return the result String.
     */
    public final String enterEqual() {
        _curState.enterEqual(this);
        System.out.println("Infix Calc enter equal: post display " + _display);
        return _display;
    }
    
    /**
     * Set this Calculator to the "clear" state where all internal memories are
     * set to zero and the pending operation is a no-op.
     */
    public final String clear() {
        _acc = 0;
        _curState = StartState.Singleton;
        _pendingOp = _noOp;
        _display = "0";
        //        System.out.println("Clear All!");
        return _display;
    }
    
    /**
     * Gets the current accumulator value
     * @return The current accumulator value
     */
    final double getAcc() {
        return _acc;
    }
    
    /**
     * Sets the accumulator value
     * @param a The new accumulator value
     */
    final void setAcc(double a) {
        _acc = a;
    }
    
    /**
     * Sets the display string
     * @param d The new display string
     */
    final void setDisplay(String d) {
        _display = d;
    }
    
    /**
     * Get the current display string
     * @return The current display string
     */
    final String getDisplay() {
        return _display;
    }
    
    /**
     * Sets the pending operation
     * @param op The new pending operation
     */
    final void setPendingOp(IBinOp op) {
        _pendingOp = op;
    }
    
    /**
     * Get the pending operation
     * 
     * @return The pending operation
     */
    final IBinOp getPendingOp() {
        return _pendingOp;
    }
    
    /**
     * Sets the pending operation to a no-op
     */
    final void setNoOp() {
        _pendingOp = _noOp;
    }
    
    /**
     * Sets the current state.
     * @param curState The new current state 
     */
    final void setCurState(ACalcState curState) {
        _curState = curState;
    }
    
    /**
     * Get the current state
     * @return The current state
     */
    final ACalcState getCurState() {
        return _curState;
    }
}

